﻿<#
.SYNOPSIS
	This script performs the deployment of a package.
.DESCRIPTION

.PARAMETER DeploymentType
	[PSADT] The type of deployment to perform. Default is: Install.
.PARAMETER DeployMode
	[PSADT] Specifies whether the installation should be run in Interactive, Silent, or NonInteractive mode. Default is: Interactive. Options: Interactive = Shows dialogs, Silent = No dialogs, NonInteractive = Very silent, i.e. no blocking apps. NonInteractive mode is automatically set if it is detected that the process is not user interactive.
.PARAMETER AllowRebootPassThru
	[PSADT] Allows the 3010 return code (requires restart) to be passed back to the parent process (e.g. SCCM) if detected from an installation. If 3010 is passed back to SCCM, a reboot prompt will be triggered.
.PARAMETER TerminalServerMode
	[PSADT] Changes to "user install mode" and back to "user execute mode" for installing/uninstalling applications for Remote Destkop Session Hosts/Citrix servers.
.PARAMETER DisableLogging
	[PSADT] Disables logging to file for the script. Default is: $false.
.PARAMETER InstallMode
	The type of installation to perform. Default is: Install.
	Derived from NI-InstallModes: imUserPart, imWkStaPart, imAppRepair, imReinstall, imModify, imUpdate, imUninstall
.PARAMETER PackagePath
	The path to the package directory or the path to the package information file. Default is: script directory.
.PARAMETER LogFileName
	The name of the log file.
.PARAMETER LogDirectory
	The directory of the log files.
.PARAMETER UserPartHandler
	User-Part handler. Default is: ActiveSetup.
.PARAMETER PersistCache
	Create a package cache entry from the content of the package.
.PARAMETER RunInInteractiveSession
	Restart process to run in interactive session. <number> = specified session; <process name> = session of the specified process; empty, -1 = current user/console session
.PARAMETER RestartInteractive
	Switch to restart process in interactive session - is equivalent to '-RunInInteractiveSession -1'
.PARAMETER InstallationParameters
	User defined installation parameters - format: <name>=<value>[, ...]. In Script.ps1 available as ${Installationparameters.<name>}
.PARAMETER IncludeForUserPart
	List of parameter names to include into the command line for immediate user part execution via separate process and into the command line for Active Setup. Default is: InstallationParameters.
.PARAMETER ExcludeForUserPart
	List of parameter names to exclude from the command line for immediate user part execution via separate process and from the command line for Active Setup. Default is: InstallMode, UserPartHandler, RunInInteractiveSession, RestartInteractive, IncludeForUserPart, ExcludeForUserPart.
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Package.ps1' -DeployMode 'Silent'; Exit $LastExitCode }"
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Package.ps1' -AllowRebootPassThru; Exit $LastExitCode }"
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Package.ps1' -DeploymentType 'Uninstall'; Exit $LastExitCode }"
.EXAMPLE
    Deploy-Application.exe -DeploymentType "Install" -DeployMode "Silent"
.NOTES
	Toolkit Exit Code Ranges:
	60000 - 68999: Reserved for built-in exit codes in Deploy-Application.ps1, Deploy-Application.exe, and AppDeployToolkitMain.ps1
	69000 - 69999: Recommended for user customized exit codes in Deploy-Application.ps1
	70000 - 79999: Recommended for user customized exit codes in AppDeployToolkitExtensions.ps1
.LINK 
	https://www.cancom.de
#>
[CmdletBinding()]
param (	
	# PSADT:
	
	[Parameter(Mandatory=$false)][ValidateSet('Install','Uninstall','Repair')]
	[string]$DeploymentType = 'Install',
	
	[Parameter(Mandatory=$false)][ValidateSet('Interactive','Silent','NonInteractive')]
	[string]$DeployMode = 'Interactive',
	
	[Parameter(Mandatory=$false)][switch]$AllowRebootPassThru = $false,
	[Parameter(Mandatory=$false)][switch]$TerminalServerMode = $false,
	[Parameter(Mandatory=$false)][switch]$DisableLogging = $false,
	
	# Deploy-Package:
	
	[Parameter(Mandatory=$false)][ValidateSet('InstallUserPart','InstallComputerPart','Install','Repair','Reinstall','Modify','Update','Uninstall')]
	[string]$InstallMode = 'Install',

	[Parameter(Mandatory=$false)][string]$PackagePath = $null,
	[Parameter(Mandatory=$false)][string]$LogFileName = $null,
	[Parameter(Mandatory=$false)][string]$LogDirectory = $null,
	[Parameter(Mandatory=$false)][string]$UserPartHandler = 'ActiveSetup',
	[Parameter(Mandatory=$false)][switch]$PersistCache = $false,

	[Parameter(Mandatory=$false)][Alias("RunInSession")][string]$RunInInteractiveSession = $null,
	[Parameter(Mandatory=$false)][switch]$RestartInteractive = $false,
	[Parameter(Mandatory=$false)][string[]]$InstallationParameters = @(),
	[Parameter(Mandatory=$false)][string[]]$IncludeForUserPart = @('InstallationParameters'),
	[Parameter(Mandatory=$false)][string[]]$ExcludeForUserPart = @('InstallMode','UserPartHandler','RunInInteractiveSession','RestartInteractive','IncludeForUserPart','ExcludeForUserPart')
)

## throw Write-Error etc.
$ErrorActionPreference = "Stop"

function Get-LocalText([string]$localizedString)
{
	$text = $localizedString.Trim();

	$matches = [regex]::Matches($localizedString, "\{(?<culture>\w+)\}\s+(?<text>[^\{]+)");
	if ($matches.Count -gt 0)
	{
		$found = @($matches | where { $_.Groups["culture"].Value -eq [cultureinfo]::CurrentUICulture.TwoLetterISOLanguageName });
		if ($found.Count -eq 0) { $found = @($matches | where { $_.Groups["culture"].Value -eq "en" }); }
		$match = $(if ($found.Count -gt 0) { $found[0] } else { $matches[0] });

		$text = $match.Groups["text"].Value.Trim();
	}

	return $text;
}

function ConvertTo-Boolean([string]$value)
{
	[bool]$flag = 0;
	if ([bool]::TryParse($value, [ref]$flag)) { return $flag; }

	[int]$number = 0;
	if ([int]::TryParse($value, [ref]$number)) { return [bool]$number; }

	return [bool]$value;
}

function Get-PackageInfo([string]$path)
{
	$path = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($path);

	$package = @{};

	$ps1Path = [System.IO.Path]::ChangeExtension($path, ".ps1");
	if ([System.IO.File]::Exists($ps1Path)) { $package = . $ps1Path; }

	$xmlPath = [System.IO.Path]::ChangeExtension($path, ".xml");
	if (![System.IO.File]::Exists($xmlPath)) { return $package; }

	$xml = New-Object System.Xml.XmlDocument;
	$xml.Load($xmlPath);

	$baseProperties = @("ID", "Name", "DisplayName", "Description");
	$node = $xml.DocumentElement.SelectSingleNode("BaseProperties");
	if ($node -ne $null) { $baseProperties = @(([string]$node.InnerText).Split(",") | % { $_.Trim() } | where { ![string]::IsNullOrEmpty($_) }); }

	$localizedProperties = @("DisplayName", "Description");
	$booleanProperties = @("HasUserPart");

	$nodes = @($baseProperties | % { $xml.DocumentElement.SelectSingleNode($_) } | where { $_ -ne $null });
	$nodes += $xml.DocumentElement.SelectNodes("Properties/Property");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });
		[string]$value = $node.InnerText;

		$isLocalizable = $(if ($node.HasAttribute("IsLocalizable")) { ConvertTo-Boolean ($node.GetAttribute("IsLocalizable")) } else { $localizedProperties -contains $name });
		if ($isLocalizable) { [string]$value = Get-LocalText $value; }

		$type = $(if ($node.HasAttribute("Type")) { try { Invoke-Expression "[$($node.GetAttribute('Type'))]" } catch { $null } } else { $null });
		if (($type -eq [bool]) -or ($booleanProperties -contains $name)) { [bool]$value = ConvertTo-Boolean $value; }
		elseif ($type -eq [int]) { $number = 0; if ([int]::TryParse($value, [ref]$number)) { [int]$value = $number; } }

		$package[$name] = $value;
	}

	$parameters = @{}
	$package["Parameters"] = $parameters;

	$nodes = $xml.DocumentElement.SelectNodes("Parameters/Parameter");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });
		[string]$value = $node.InnerText;
		if (($node.FirstChild -ne $null) -and ($node.FirstChild.NodeType -eq "Text")) { $value = $node.FirstChild.InnerText; }
		elseif ($node.HasAttribute("Value")) { $value = $node.GetAttribute("Value"); }
		elseif ($node.HasAttribute("DefaultValue")) { $value = $node.GetAttribute("DefaultValue"); }

		$isLocalizable = $(if ($node.HasAttribute("IsLocalizable")) { ConvertTo-Boolean ($node.GetAttribute("IsLocalizable")) } else { $localizedProperties -contains $name });
		if ($isLocalizable) { [string]$value = Get-LocalText $value; }

		$type = $(if ($node.HasAttribute("Type")) { try { Invoke-Expression "[$($node.GetAttribute('Type'))]" } catch { $null } } else { $null });
		if (($type -eq [bool]) -or ($booleanProperties -contains $name)) { [bool]$value = ConvertTo-Boolean $value; }
		elseif ($type -eq [int]) { $number = 0; if ([int]::TryParse($value, [ref]$number)) { [int]$value = $number; } }

		$parameters[$name] = $value;
	}

	$xmlDefaultsPath = [System.IO.Path]::Combine([System.IO.Path]::GetDirectoryName($path), "Package.Defaults.xml");
	if (![System.IO.File]::Exists($xmlDefaultsPath)) { return $package; }

	$xmlDefaults = New-Object System.Xml.XmlDocument;
	$xmlDefaults.Load($xmlDefaultsPath);

	$nodes = @($baseProperties | % { $xmlDefaults.DocumentElement.SelectSingleNode($_) } | where { $_ -ne $null });
	$nodes += $xmlDefaults.DocumentElement.SelectNodes("Properties/Property");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });
		if ($package.ContainsKey($name)) { continue; }

		[string]$value = $node.InnerText;

		$isLocalizable = $(if ($node.HasAttribute("IsLocalizable")) { ConvertTo-Boolean ($node.GetAttribute("IsLocalizable")) } else { $localizedProperties -contains $name });
		if ($isLocalizable) { [string]$value = Get-LocalText $value; }

		$type = $(if ($node.HasAttribute("Type")) { try { Invoke-Expression "[$($node.GetAttribute('Type'))]" } catch { $null } } else { $null });
		if (($type -eq [bool]) -or ($booleanProperties -contains $name)) { [bool]$value = ConvertTo-Boolean $value; }
		elseif ($type -eq [int]) { $number = 0; if ([int]::TryParse($value, [ref]$number)) { [int]$value = $number; } }

		$package[$name] = $value;
	}

	$nodes = $xmlDefaults.DocumentElement.SelectNodes("Parameters/Parameter");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });
		if ($parameters.ContainsKey($name)) { continue; }

		[string]$value = $node.InnerText;
		if (($node.FirstChild -ne $null) -and ($node.FirstChild.NodeType -eq "Text")) { $value = $node.FirstChild.InnerText; }
		elseif ($node.HasAttribute("Value")) { $value = $node.GetAttribute("Value"); }
		elseif ($node.HasAttribute("DefaultValue")) { $value = $node.GetAttribute("DefaultValue"); }

		$isLocalizable = $(if ($node.HasAttribute("IsLocalizable")) { ConvertTo-Boolean ($node.GetAttribute("IsLocalizable")) } else { $localizedProperties -contains $name });
		if ($isLocalizable) { [string]$value = Get-LocalText $value; }

		$type = $(if ($node.HasAttribute("Type")) { try { Invoke-Expression "[$($node.GetAttribute('Type'))]" } catch { $null } } else { $null });
		if (($type -eq [bool]) -or ($booleanProperties -contains $name)) { [bool]$value = ConvertTo-Boolean $value; }
		elseif ($type -eq [int]) { $number = 0; if ([int]::TryParse($value, [ref]$number)) { [int]$value = $number; } }

		$parameters[$name] = $value;
	}

	$internalDefaults = @{ "DisplayVersion" = "Version"; };
	$internalDefaults.GetEnumerator() | where { $package.ContainsKey($_.Key) -and [string]::IsNullOrEmpty($package[$_.Key]) -and $package.ContainsKey($_.Value) } | % { $package[$_.Key] = $package[$_.Value]; }

	return $package;
}

function Merge-ParameterValues($package, [string[]]$parameterValues = @())
{
	if (!($package -is [hashtable])) { return; }

	$parameters = $package["Parameters"];
	if ($parameters -eq $null) { return; }

	foreach ($item in $parameterValues)
	{
		if ([string]::IsNullOrWhiteSpace($item)) { continue; }
		
		$index = $item.IndexOfAny(":="); # <name>=<value> or <name>:<value>
		$name = $(if ($index -ge 0) { $item.Substring(0, $index).Trim() } else { $item.Trim() });
		$value = $(if ($index -ge 0) { $item.Substring($index + 1) } else { $null });
		if (!$parameters.ContainsKey($name)) { continue; }

		$match = [regex]::Match($value, "^b(ase)?64[:=]\s*(?<value>[^\s]*)\s*`$", "IgnoreCase");
		if ($match.Success) { $value = $(try { [System.Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($match.Groups["value"].Value)) } catch { $value }); }

		$parameters[$name] = $value;
	}
}

function ConvertTo-ArgumentString([hashtable]$parameters, [string[]]$include = @(), [string[]]$exclude = @(), [switch]$forScript)
{
	$forExe = !$forScript;

	$items = @();

	foreach ($kvp in $parameters.GetEnumerator())
	{
		$name = $kvp.Key;
		$value = $kvp.Value;

		if (($include.Count -gt 0) -and ($include -notcontains $name)) { continue; }
		if (($exclude.Count -gt 0) -and ($exclude -contains $name)) { continue; }

		$item = $null;

		if ($value -eq $null) { $item = "-$($name):`$null"; }
		elseif ($value -is [switch]) { $item = $(if ($value) { "-$($name)" } else { "-$($name):`$false" }); }
		elseif ($value -is [bool]) { $item = "-$($name) $(if ($value) { 1 } else { 0 })"; }
		elseif ($value -is [datetime]) { $item = "-$($name) `"$($value.ToUniversalTime().ToString('u'))`""; }
		elseif (($value -is [array]) -and $forExe) { $list = @($value | % { $text = $_.ToString([cultureinfo]::InvariantCulture); $(if ($text.IndexOf(" ") -ge 0) { "'$($text)'"; } else { $text }); }); $text = [string]::Join(', ', $list); $item = $(if ($list.Count -eq 0) { "" } elseif ($list.Count -eq 1) { "-$($name) $($text)" } else { "-$($name) `"$($text)`"" }); }
		elseif (($value -is [array]) -and $forScript) { $list = @($value | % { $text = $_.ToString([cultureinfo]::InvariantCulture); $(if ($text.IndexOf(" ") -ge 0) { "'$($text)'"; } else { $text }); }); $text = [string]::Join(', ', $list); $item = $(if ($list.Count -eq 0) { "-$($name) @()" } else { "-$($name) $($text)" }); }
		else { $text = $value.ToString([cultureinfo]::InvariantCulture); if ($text.IndexOf(" ") -ge 0) { $text = "`"$($text)`""; }; $item = "-$($name) $($text)"; }

		if ([string]::IsNullOrEmpty($item)) { continue; }

		$items += $item;
	}
	
	return [string]::Join(" ", $items);
}

try {	
	Push-Location;

	## Set ExecutionPolicy ByPass for process
	try { Set-ExecutionPolicy -ExecutionPolicy ByPass -Scope Process -Force -ErrorAction Stop } catch {}
	
	##*===============================================
	##* VARIABLE DECLARATION
	##*===============================================
	$context = @{

		## Variables: Install Titles (Only set here to override defaults set by the toolkit)
		installName = ''
		installTitle = ''
		
		## Variables: Script
		deployAppScriptFriendlyName = 'Deploy Package'
		deployAppScriptVersion = [version]'1.0.0'
		deployAppScriptDate = ''
		deployAppScriptParameters = $psBoundParameters
		
		## PSADT: exported globals - e.g.: 'env*','appDeploy*'
		adtVariables = $null
		adtFunctions = $null
	}
		
	## Variables: Exit Code
	[int32]$mainExitCode = 0

	## Parameters
	if ($PSBoundParameters.ContainsKey("InstallMode"))
	{
		if ($InstallMode -eq "Uninstall") { $DeploymentType = "Uninstall" }
		elseif ($InstallMode -eq "Repair") { $DeploymentType = "Repair" }
		elseif (($InstallMode -eq "InstallUserPart") -and ($UserPartHandler -match "^(ActiveSetup|InternalSetup)\.Uninstall`$")) { $DeploymentType = "Uninstall" }
		elseif (($InstallMode -eq "InstallUserPart") -and ($UserPartHandler -match "^(ActiveSetup|InternalSetup)\.Repair`$")) { $DeploymentType = "Repair" }
		else { $DeploymentType = "Install" }
	}
	elseif ($PSBoundParameters.ContainsKey("DeploymentType"))
	{
		$InstallMode = $DeploymentType;
	}
	
	$context.DeploymentType = $DeploymentType;
	$context.DeployMode = $DeployMode;
	$context.AllowRebootPassThru = $AllowRebootPassThru;
	$context.TerminalServerMode = $TerminalServerMode;
	$context.DisableLogging = $DisableLogging;
	$context.InstallMode = $InstallMode;
	$context.ScriptPath = $null;
	$context.ContinueScript = $null;
	$context.LogFileName = $LogFileName;
	$context.LogDirectory = $LogDirectory;
	$context.UserPartHandler = $UserPartHandler;
	
	## Variables: Environment
	If (Test-Path -LiteralPath 'variable:HostInvocation') { $InvocationInfo = $HostInvocation } Else { $InvocationInfo = $MyInvocation }
	[string]$scriptDirectory = Split-Path -Path $InvocationInfo.MyCommand.Definition -Parent
	[string]$scriptName = Split-Path -Path $InvocationInfo.MyCommand.Definition -Leaf
	
	## Context: Package Information
	$packageDirectory = $null;
	$packageInfoPath = $null;
	if ([string]::IsNullOrEmpty($PackagePath)) { $PackagePath = $scriptDirectory; } else { $PackagePath = (Resolve-Path $PackagePath).ProviderPath; }
	if (Test-Path $PackagePath -PathType Container)
	{
		$packageDirectory = $PackagePath;
		$packageInfoPath = Join-Path -Path $packageDirectory -ChildPath Package.xml;
	}
	else # (Test-Path $PackagePath -PathType Leaf)
	{
		$packageInfoPath = $PackagePath;
		$packageDirectory = Split-Path -Path $packageInfoPath -Parent;
	}
	
	$package = Get-PackageInfo -path $packageInfoPath;
	Merge-ParameterValues -package $package -parameterValues $InstallationParameters;

	$context.PackageDirectory = $packageDirectory;
	$context.Package = $package;

	$context.InstallTitle = "$($package.Vendor) $($package.Name) $(if (![string]::IsNullOrEmpty($package.Version) -and (([string]$package.Name).IndexOf($package.Version) -lt 0)) {$package.Version})".Trim();
	
	${LogSource} = "$($scriptName)/$($InstallMode)";
	if (![String]::IsNullOrEmpty($package.Name) -and [String]::IsNullOrEmpty($context.LogFileName))
	{
		$timestamp = [DateTime]::Now.ToString("yyyy-MM-dd-HH-mm-ss-fff");
		$format = $(if (![String]::IsNullOrEmpty($package.LogFileNameFormat)) { $package.LogFileNameFormat }  else { '$($package.Vendor)_$($package.Name)_$($package.Version)_$($InstallMode)_$($timestamp).log' });
		$context.LogFileName = Invoke-Expression "`"$($format)`"";
	}
	
	## Import the PackageDeployment module
	$PackageDeploymentModule = "PackageDeployment";
	try
	{
		if (Test-Path -PathType Container -Path "$($scriptDirectory)\PackageDeployment") { $PackageDeploymentModule = "$($scriptDirectory)\PackageDeployment"; }
		elseif (Test-Path -PathType Leaf -Path "$($scriptDirectory)\PackageDeployment.psd1") { $PackageDeploymentModule = "$($scriptDirectory)\PackageDeployment.psd1"; }
		Import-Module $PackageDeploymentModule -Force -ArgumentList $context;
	}
	catch
	{
		If ($mainExitCode -eq 0){ [int32]$mainExitCode = 60008 }
		Write-Error -Message "Module [$PackageDeploymentModule] failed to load: `n$($_.Exception.Message)`n `n$($_.InvocationInfo.PositionMessage)" -ErrorAction 'Continue'
		## Exit the script, returning the exit code to SCCM
		If (Test-Path -LiteralPath 'variable:HostInvocation') { $script:ExitCode = $mainExitCode; Exit } Else { Exit $mainExitCode }
	}

	## restart interactive
	$restartedProcess = $null;
	if ($RestartInteractive -or $PSBoundParameters.ContainsKey("RunInInteractiveSession"))
	{
		if ($RestartInteractive) { Write-Log -Message "Parameter -RestartInteractive is set, -RunInInteractiveSession is '$($RunInInteractiveSession)'." -Source ${LogSource} }
		else { Write-Log -Message "Parameter -RunInInteractiveSession is set to '$($RunInInteractiveSession)'." -Source ${LogSource} }

		$currentProcess = [System.Diagnostics.Process]::GetCurrentProcess();
		Write-Log -Message "Current process: '$($currentProcess.ProcessName)' with ID $($currentProcess.Id) in session $($currentProcess.SessionId)." -Source ${LogSource}

		$PSPD_RestartedFrom = "PSPD_RestartedFrom";
		$restartedFromProcessID = [System.Environment]::GetEnvironmentVariable($PSPD_RestartedFrom);
		if (![string]::IsNullOrEmpty($restartedFromProcessID))
		{
			Write-Log -Message "Continue in current process: detected environment variable '$($PSPD_RestartedFrom)' (current process was restarted from process with ID $($restartedFromProcessID))." -Source ${LogSource}
		}
		elseif ([PSPD.API]::GetActiveSessionIDs() -contains $currentProcess.SessionId)
		{
			Write-Log -Message "Continue in current process: already running in interactive session $($currentProcess.SessionId)." -Source ${LogSource}
		}
		else
		{
			try
			{
				Write-Log -Message "Restarting the process in the current interactive user session." -Source ${LogSource}

				Write-Log -Message "Sessions:`r`n$(([PSPD.API]::GetSessionInfos() | ft @{l='Session ID';e={$_.SessionID}}, @{l='WinStation name';e={$_.pWinStationName}}, @{l='Connection state';e={$_.State}} -AutoSize | Out-String).Trim())" -Source ${LogSource}

				[System.Environment]::SetEnvironmentVariable($PSPD_RestartedFrom, $currentProcess.Id);
				$restartedProcess = [PSPD.API]::RestartProcessInteractive($RunInInteractiveSession);
				if ($restartedProcess -ne $null)
				{
					Write-Log -Message "Restarted process: '$($restartedProcess.ProcessName)' with ID $($restartedProcess.Id) in session $($restartedProcess.SessionId)." -Source ${LogSource}
					$restartedProcess.WaitForExit();

					Write-Log -Message "The restarted process '$($restartedProcess.ProcessName)' with ID $($restartedProcess.Id) returned: $($restartedProcess.ExitCode)." -Source ${LogSource}
					return; # -> continue in finally-block
				}
				else
				{
					Write-Log -Message "Continue in current process: restart not required." -Source ${LogSource}
				}
			}
			catch [System.OperationCanceledException]
			{
				Write-Log -Message "Continue in current process: $($_.ToString())" -Source ${LogSource}
			}
			catch
			{
				$failed = "Restart the process in interactive mode";
				Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source $LogSource;
				Write-Log -Message "Continue in current process: failed to restart process." -Source ${LogSource}
				# throw $_;
			}
		}
	}

	## Import the ShowMessage module
	$ShowMessageModule = "ShowMessage";
	try
	{
		if (Test-Path -PathType Container -Path "$($scriptDirectory)\SupportFiles") { $ShowMessageModule = "$($scriptDirectory)\SupportFiles\ShowMessage.dll"; }
		Import-Module $ShowMessageModule -Force;
	}
	catch
	{
		Write-Log -Message "Module [$ShowMessageModule] failed to load: `n$($_.Exception.Message)`n `n$($_.InvocationInfo.PositionMessage)" -Severity 2 -Source ${LogSource}
	}
	
	$PdContext = Get-PdContext;
	$PdContext.InstallPhase = "$($scriptName)/$($InstallMode)";
	$PdContext.PackageControlsReboot = $true; # allow the package to reboot the computer

	$executeComputerPart = ($PdContext.InstallMode -ne "InstallUserPart");
	$hasUserPart = [bool]$PdContext.Package.HasUserPart;
	$skipUserPart = [bool]$PdContext.Package.SkipImmediateUserPartExecution;
	$executeUserPart = ($hasUserPart -and !$skipUserPart -and ($PdContext.DeployMode -eq "Interactive") -and ($PdContext.InstallMode -ne "InstallComputerPart"));
    $PdContext.IncludeUserPart = ($executeUserPart -and !$PdContext.SessionZero);
    $script:restartForUserPart = ($executeUserPart -and $PdContext.SessionZero);
	$mode = [regex]::Match([string]$PdContext.UserPartHandler, "^(ActiveSetup|InternalSetup)\.(?<mode>.+)").Groups["mode"].Value;
	if ([string]::IsNullOrEmpty($mode)) { $mode = $PdContext.InstallMode; }
	$PdContext.UserPartInstallMode = $(if (($mode -eq "InstallComputerPart") -or ($mode -eq "InstallUserPart")) { "Install" } else { $mode });
	$hideBalloonNotifications = ($PdContext.ShowBalloonNotifications -and ($PdContext.UserPartHandler -match "^InternalSetup(\.|`$)"));
	if ($hideBalloonNotifications) { $PdContext.ShowBalloonNotifications = $false; }
	$showFinalErrorMessage = [bool]$PdContext.Package.ShowFinalErrorMessage;

	## Log package deployment information
	$infoSeparator = '#' * 79
	Write-Log -Message $infoSeparator -Source ${LogSource}
	$currentProcess = [System.Diagnostics.Process]::GetCurrentProcess()
	Write-Log -Message "Current process ID: $($currentProcess.Id)" -Source ${LogSource}
	Write-Log -Message "Current process session ID: $($currentProcess.SessionId)" -Source ${LogSource}
	Write-Log -Message "Current process is 64 Bit: $([System.Environment]::Is64BitProcess)" -Source ${LogSource}
	Write-Log -Message "Current process is elevated: $($PdContext.UserIsAdmin)" -Source ${LogSource}
	Write-Log -Message "Executing user account: $($PdContext.User.Process)" -Source ${LogSource}
	Write-Log -Message "Executing user is service account (Context.SessionZero): $($PdContext.SessionZero)" -Source ${LogSource}
	Write-Log -Message "Executing user account: $($PdContext.User.Process)" -Source ${LogSource}
	Write-Log -Message "Logged on user account: $($PdContext.User.LoggedOn)" -Source ${LogSource}
	Write-Log -Message "Package name: $($PdContext.Package.Name)" -Source ${LogSource}
	Write-Log -Message "Package version: $($PdContext.Package.Version)" -Source ${LogSource}
	Write-Log -Message "Package revision: $($PdContext.Package.Revision)" -Source ${LogSource}
	Write-Log -Message "DeployMode: $($PdContext.DeployMode)" -Source ${LogSource}
	Write-Log -Message "DeploymentType: $($PdContext.DeploymentType)" -Source ${LogSource}
	Write-Log -Message "InstallMode: $($PdContext.InstallMode)" -Source ${LogSource}
	Write-Log -Message "Package has user part (Package.HasUserPart): $($hasUserPart)" -Source ${LogSource}
	Write-Log -Message "UserPartHandler: $($PdContext.UserPartHandler)" -Source ${LogSource}
	Write-Log -Message "UserPartInstallMode: $($PdContext.UserPartInstallMode)" -Source ${LogSource}
	Write-Log -Message "Skip immediate user part execution (Package.SkipImmediateUserPartExecution): $($skipUserPart)" -Source ${LogSource}
	Write-Log -Message "Executing computer part: $($executeComputerPart)" -Source ${LogSource}
	$infos = @(if ($PdContext.IncludeUserPart) { if ($executeComputerPart) { "together with computer part" } } elseif ($restartForUserPart) { "via separate process" } else { if ($skipUserPart) { "Package.SkipImmediateUserPartExecution" }; if ($PdContext.DeployMode -ne "Interactive") { "DeployMode $($PdContext.DeployMode)" }; if ($PdContext.InstallMode -eq "InstallComputerPart") { "InstallMode $($PdContext.InstallMode)" } });
	$userPartInfo = $(if ($infos.Count -gt 0) { " ($([string]::Join(', ', $infos)))" } else { $null });
	Write-Log -Message "Executing user part: $($executeUserPart)$($userPartInfo)" -Source ${LogSource}
	if ($hideBalloonNotifications) { Write-Log -Message "Hiding balloon notifications: $(!$PdContext.ShowBalloonNotifications)" -Source ${LogSource}; }
	Write-Log -Message "Installed Apps registration: $($PdContext.InstalledAppsRegistryKey)" -Source ${LogSource}
	Write-Log -Message $infoSeparator -Source ${LogSource}

	## Perform package scripts
	$packageScripts = @();
	if (!$package.ContainsKey("Script") -or [string]::IsNullOrEmpty($package.Script)) { $packageScripts = @("Script.ps1"); }
	elseif ($package.Script -is [string]) { $packageScripts = @($package.Script); }
	elseif ($package.Script -is [string[]]) { $packageScripts = $package.Script; }

	$packageScripts = @($packageScripts | % { [regex]::Replace($_, "(^[\.\\/]+[\\/]|^[\\/]+|[\\/]+\.+(?=[\\/]+))", "") });
	
	$gotoPs1Path = "$($packageDirectory)\goto.ps1";
	
	Write-Log -Message "Processing scripts ($($packageScripts.Count)): $([string]::Join(' ', $packageScripts))." -Source ${LogSource}
	foreach ($script in $packageScripts)
	{
		$PdContext.InstallPhase = "$($script)/$($InstallMode)";
		$scriptPath = "$($packageDirectory)\$($script)"
		if (Test-Path $scriptPath)
		{
			Set-Location $packageDirectory;
			Write-Log -Message "Current location is: '$((Get-Location).Path)'." -Source ${LogSource}
			[System.Environment]::CurrentDirectory = (Get-Location -PSProvider FileSystem).ProviderPath;
			Write-Log -Message "Working directory is: '$([System.Environment]::CurrentDirectory)'." -Source ${LogSource}
			
			$PdContext.ScriptPath = $scriptPath;
			$PdContext.ContinueScript = $null;
			Write-Log -Message "Processing script '$($scriptPath)' for mode '$($InstallMode)'." -Source ${LogSource}
			Clear-DeleteAtEndOfScript;

			. $scriptPath
			
			while ($PdContext.ContinueScript -is [scriptblock])
			{
				$continueScript = $PdContext.ContinueScript;

				if (Test-Path $gotoPs1Path)
				{
					$PdContext.ContinueScript = $continueScript.ToString();

					Write-Log -Message "Continue with goto script ($($gotoPs1Path)) ..." -Source ${LogSource}
					. $gotoPs1Path;
				}
				else
				{
					$PdContext.ContinueScript = $null;
					$tmpPath = [System.IO.Path]::Combine([System.IO.Path]::GetTempPath(), "$([Guid]::NewGuid()).ps1");
					[System.IO.File]::WriteAllText($tmpPath, $continueScript.ToString(), [System.Text.Encoding]::UTF8);
					
					Write-Log -Message "Continue script '$($tmpPath)' ..." -Source ${LogSource}
					. $tmpPath;

					try { [System.IO.File]::Delete($tmpPath); } catch {}
				}
			}

			if ([string]::IsNullOrEmpty($LogName))
			{
				$global:LogName = $LogFileName;
				Write-Log -Message "Script has finished and the log file name is empty. Restored the log file name to '$($LogName)'." -Source ${LogSource};

				if ([string]::IsNullOrEmpty($InstallName))
				{
					$global:InstallName = $package.Name;
					Write-Log -Message "Restored the installation name to '$($InstallName)'." -Source ${LogSource};
				}

				if ($LASTEXITCODE -is [int])
				{
					# e.g. via Show-InstallationPrompt (-ExitOnTimeout), Execute-Process (-ExitOnProcessFailure), Show-InstallationWelcome (timeout, defer, disk space requirements), or caused by an assembly load failure. 
					Write-Log -Message "Maybe there was an explicit call to Exit-Script, which set the last exit code to $($LASTEXITCODE)." -Source ${LogSource};

					if (($mainExitCode -eq 0) -and ($mainExitCode -ne $LASTEXITCODE))
					{
						$mainExitCode = $LASTEXITCODE;
						Write-Log -Message "Setting the main exit code to the last exit code ($($mainExitCode))." -Source ${LogSource};
					}
					elseif ($mainExitCode -ne 0)
					{
						Write-Log -Message "The current main exit code is $($mainExitCode) - ignoring the last exit code ($($LASTEXITCODE))." -Source ${LogSource};
					}
					
					if ($global:configShowBalloonNotifications)
					{
						$global:configShowBalloonNotifications = $false;
						Write-Log -Message "Disabled balloon notifications for the final Exit-Script call." -Source ${LogSource};
					}
				}
			}

			Invoke-DeleteAtEndOfScript
			
			Write-Log -Message "Finished script '$($scriptPath)'. Reboot requests: $($PdContext.RebootRequests)" -Source ${LogSource}
			
			if (![string]::IsNullOrEmpty($PdContext.Status))
			{
				switch ($PdContext.Status)
				{
					"Done" { $mainExitCode = 0; break; }
					"Undone" { $mainExitCode = $([int]$n = 0; if ([int]::TryParse($PdContext.Package.ExitCodeOnPackageStatusUndone, [ref]$n)) {$n} else {69000}); break; }
					"UndoneContinueParentScript" { $mainExitCode = 69001; break; }
					"Failed" { $mainExitCode = 69002; break; }
					default { throw "Unknown exit status code '$($PdContext.Status)'."; }
				}
				
				$status = "'$($PdContext.Status)'"; if (![string]::IsNullOrEmpty($PdContext.StatusMessage)) { $status += " [$($PdContext.StatusMessage)]"; }
				Write-Log -Message "Exit status is $($status) - setting exit code to $($mainExitCode)." -Source ${LogSource}
				break; # -> exit foreach
			}
			else
			{
				if (($mainExitCode -eq $PdContext.PSADT.InstallationDeferExitCode) -and ![string]::IsNullOrEmpty($PDContext.Package.PdStatusOnInstallationDeferExitCode))
				{
					$PdContext.Status = [string]$PDContext.Package.PdStatusOnInstallationDeferExitCode;
					Write-Log -Message "Exit code is $($mainExitCode) - setting exit status to $($PdContext.Status)." -Source ${LogSource}
				}
				elseif (($mainExitCode -eq $PdContext.PSADT.InstallationUIExitCode) -and ![string]::IsNullOrEmpty($PDContext.Package.PdStatusOnInstallationUIExitCode))
				{
					$PdContext.Status = [string]$PDContext.Package.PdStatusOnInstallationUIExitCode;
					Write-Log -Message "Exit code is $($mainExitCode) - setting exit status to $($PdContext.Status)." -Source ${LogSource}
				}
			}
		}
		else
		{
			throw "Script '$($scriptPath)' for mode '$($InstallMode)' not present."
		}
	}

	# MSI: 3010 = ERROR_SUCCESS_REBOOT_REQUIRED, 1641 = ERROR_SUCCESS_REBOOT_INITIATED
	if (($mainExitCode -eq 0) -and ([int]$PdContext.RebootRequests -gt 0))
	{
		$mainExitCode = 3010;

		if ($PdContext.StartSystemShutdownParameters -ne $null)
		{
			$parameters = $PdContext.StartSystemShutdownParameters;
			$action = $(if ($parameters.ContainsKey("Restart")) { "reboot" } else { "shutdown" });
			Write-Log -Message "The package requested a $($action) of the computer (request count: $($PdContext.RebootRequests))." -Source ${LogSource}

			if ($PdContext.PackageControlsReboot)
			{
				Write-Log -Message "Initializing the requested $($action)." -Source ${LogSource}
				$success = Start-SystemShutdown -PassThru @parameters;
				if ($success) { $mainExitCode = 1641; }
			}
			else
			{
				Write-Log -Message "The package is not allowed to shut down or restart the computer." -Source ${LogSource}
			}
		}
		
		Write-Log -Message "Reboot requests: $($PdContext.RebootRequests) - setting exit code to $($mainExitCode)." -Source ${LogSource}
	}
}
catch
{
	[int32]$mainExitCode = 60001
	[string]$mainErrorMessage = "$(Resolve-Error)"
	Write-Log -Message $mainErrorMessage -Severity 3 -Source ${LogSource}

	if ([string]::IsNullOrEmpty($PdContext.Status) -and ![string]::IsNullOrEmpty($PDContext.Package.PdStatusOnException))
	{
		$PdContext.Status = [string]$PDContext.Package.PdStatusOnException;
		$PdContext.StatusMessage = $_.Exception.Message;
		Write-Log -Message "Failed with exception: setting exit status to '$($PdContext.Status)' (PdStatusOnException) and status message to '$($PdContext.StatusMessage)'." -Source ${LogSource}
	}

	if (($DeployMode -eq "Interactive") -and $showFinalErrorMessage) { $void = Show-DialogBox -Text $mainErrorMessage -Icon "Stop" }
}
finally
{
	Pop-Location;

	if ($restartedProcess -ne $null)
	{
		# note: 
		# when finishing the "## restart interactive" sequence
		# some values may be unavailable here (e.g. $PdContext is undefined)

		Write-Log -Message "Main exit code is: $($mainExitCode)." -Source ${LogSource}

		$exitCode = $restartedProcess.ExitCode;
		Write-Log -Message "Returning the exit code from the restarted process '$($restartedProcess.ProcessName)' with ID $($restartedProcess.Id) in session $($restartedProcess.SessionId): $($exitCode)." -Source ${LogSource}
		Exit-Script -ExitCode $exitCode;
	}

	$PdContext.InstallPhase = "$($scriptName)/Finalization";

	Write-InstalledAppsRegistry -ExitCode $mainExitCode -WriteAllProperties -ContinueOnError;
	$success = ($mainExitCode -in @(0, 1641, 3010));
	
	$myCommandPath = $InvocationInfo.MyCommand.Definition;
	if ($success -and $PersistCache)
	{
		$entryDir = New-PackageCacheEntry -ScriptPath $myCommandPath -ContinueOnError;
		if (![string]::IsNullOrEmpty($entryDir)) { Write-Log -Message "Package cache entry created at '$($entryDir)'." -Source ${LogSource}; }
		else { Write-Log -Message "No package cache entry created - although -PersistCache is set." -Severity 2 -Source ${LogSource}; }
		if (![string]::IsNullOrEmpty($myCommandPath))
		{
			if (![string]::IsNullOrEmpty($entryDir)) { $myCommandPath = [System.IO.Path]::Combine($entryDir, [System.IO.Path]::GetFileName($myCommandPath)); }
			Write-Log -Message "Using script path '$($myCommandPath)'." -Source ${LogSource}
		}
	}
	
	if ($success) { Write-ActiveSetupRegistry -ScriptPath $myCommandPath -Arguments (ConvertTo-ArgumentString -parameters $PSBoundParameters -include $IncludeForUserPart -exclude $ExcludeForUserPart) -ContinueOnError; }
	else { Write-Log -Message "Skip writing Active Setup Registry: Exit code is $($mainExitCode)." -Source ${LogSource} }

	if ($success -and $script:restartForUserPart -and ($mainExitCode -eq 1641))
	{
		Write-Log -Message "Skip invoking user part installation: a system reboot was initiated (exit code $($mainExitCode))." -Source ${LogSource}
	}
	elseif ($success -and $script:restartForUserPart)
	{
		Write-Log -Message "Executing user part via separate process." -Source ${LogSource}
		$userPartExitCode = 0;

		try
		{
			$applicationName = [System.IO.Path]::Combine([System.IO.Path]::GetDirectoryName($myCommandPath), "Deploy-Application.exe");
			$optionalArguments = ConvertTo-ArgumentString -parameters $PSBoundParameters -include $IncludeForUserPart -exclude $ExcludeForUserPart;
			$arguments = "-InstallMode InstallUserPart -UserPartHandler InternalSetup.$($PdContext.InstallMode) $($optionalArguments)".Trim();

			$process = $null;
			if ($PdContext.SessionZero)
			{
				Write-Log -Message "Running '$($applicationName)' with arguments [$($arguments)] in logged-on user session." -Source ${LogSource};
				$process = [PSPD.API]::StartInteractive($applicationName, $arguments, $null, "~");
				Write-Log -Message "Started process '$($process.ProcessName)' with ID $($process.Id) in user session $($process.SessionId)." -Source ${LogSource};
			}
			else
			{
				Write-Log -Message "Running '$($applicationName)' with arguments [$($arguments)] in current session." -Source ${LogSource};
				$process = [System.Diagnostics.Process]::Start($applicationName, $arguments);
				Write-Log -Message "Started process '$($process.ProcessName)' with ID $($process.Id) in current session ($($process.SessionId))." -Source ${LogSource};
			}

			$process.WaitForExit();
			$userPartExitCode = $process.ExitCode;
			Write-Log -Message "Process '$($process.ProcessName)' with ID $($process.Id) terminated with exit code $($userPartExitCode)." -Source ${LogSource};
		}
		catch
		{
			$userPartExitCode = 60001
			$userPartErrorMessage = "$($failed).`r`n$(Resolve-Error)";
			Write-Log -Message $userPartErrorMessage -Severity 3 -Source ${LogSource}
			if ($DeployMode -eq "Interactive") { $void = Show-DialogBox -Text $userPartErrorMessage -Icon "Stop" }
		}
		
		if ($userPartExitCode -ne 0)
		{
			Write-Log -Message "Inital process exit code: $($mainExitCode)." -Source ${LogSource};
			Write-Log -Message "User part process exit code: $($userPartExitCode)." -Source ${LogSource};

			$mainExitCode = $userPartExitCode;
			Write-Log -Message "Resulting exit code: $($mainExitCode)." -Source ${LogSource};
			$success = ($mainExitCode -in @(0, 1641, 3010));
			Write-Log -Message "Resulting status: $(if ($success) {'success'} else {'failure'})." -Source ${LogSource};
		}
	}

	Exit-Script -ExitCode $mainExitCode
}

# SIG # Begin signature block
# MIIrvQYJKoZIhvcNAQcCoIIrrjCCK6oCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAyDeu1U0zjqh6I
# BCvTGrNpmCyEwCafSYB+1OFy0bOZKqCCJNMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYUMIID/KADAgECAhB6I67a
# U2mWD5HIPlz0x+M/MA0GCSqGSIb3DQEBDAUAMFcxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIFRp
# bWUgU3RhbXBpbmcgUm9vdCBSNDYwHhcNMjEwMzIyMDAwMDAwWhcNMzYwMzIxMjM1
# OTU5WjBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSww
# KgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjCCAaIw
# DQYJKoZIhvcNAQEBBQADggGPADCCAYoCggGBAM2Y2ENBq26CK+z2M34mNOSJjNPv
# IhKAVD7vJq+MDoGD46IiM+b83+3ecLvBhStSVjeYXIjfa3ajoW3cS3ElcJzkyZlB
# nwDEJuHlzpbN4kMH2qRBVrjrGJgSlzzUqcGQBaCxpectRGhhnOSwcjPMI3G0hedv
# 2eNmGiUbD12OeORN0ADzdpsQ4dDi6M4YhoGE9cbY11XxM2AVZn0GiOUC9+XE0wI7
# CQKfOUfigLDn7i/WeyxZ43XLj5GVo7LDBExSLnh+va8WxTlA+uBvq1KO8RSHUQLg
# zb1gbL9Ihgzxmkdp2ZWNuLc+XyEmJNbD2OIIq/fWlwBp6KNL19zpHsODLIsgZ+WZ
# 1AzCs1HEK6VWrxmnKyJJg2Lv23DlEdZlQSGdF+z+Gyn9/CRezKe7WNyxRf4e4bwU
# trYE2F5Q+05yDD68clwnweckKtxRaF0VzN/w76kOLIaFVhf5sMM/caEZLtOYqYad
# tn034ykSFaZuIBU9uCSrKRKTPJhWvXk4CllgrwIDAQABo4IBXDCCAVgwHwYDVR0j
# BBgwFoAU9ndq3T/9ARP/FqFsggIv0Ao9FCUwHQYDVR0OBBYEFF9Y7UwxeqJhQo1S
# gLqzYZcZojKbMA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMG
# A1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBMBgNVHR8ERTBD
# MEGgP6A9hjtodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1l
# U3RhbXBpbmdSb290UjQ2LmNybDB8BggrBgEFBQcBAQRwMG4wRwYIKwYBBQUHMAKG
# O2h0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGlu
# Z1Jvb3RSNDYucDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNv
# bTANBgkqhkiG9w0BAQwFAAOCAgEAEtd7IK0ONVgMnoEdJVj9TC1ndK/HYiYh9lVU
# acahRoZ2W2hfiEOyQExnHk1jkvpIJzAMxmEc6ZvIyHI5UkPCbXKspioYMdbOnBWQ
# Un733qMooBfIghpR/klUqNxx6/fDXqY0hSU1OSkkSivt51UlmJElUICZYBodzD3M
# /SFjeCP59anwxs6hwj1mfvzG+b1coYGnqsSz2wSKr+nDO+Db8qNcTbJZRAiSazr7
# KyUJGo1c+MScGfG5QHV+bps8BX5Oyv9Ct36Y4Il6ajTqV2ifikkVtB3RNBUgwu/m
# SiSUice/Jp/q8BMk/gN8+0rNIE+QqU63JoVMCMPY2752LmESsRVVoypJVt8/N3qQ
# 1c6FibbcRabo3azZkcIdWGVSAdoLgAIxEKBeNh9AQO1gQrnh1TA8ldXuJzPSuALO
# z1Ujb0PCyNVkWk7hkhVHfcvBfI8NtgWQupiaAeNHe0pWSGH2opXZYKYG4Lbukg7H
# pNi/KqJhue2Keak6qH9A8CeEOB7Eob0Zf+fU+CCQaL0cJqlmnx9HCDxF+3BLbUuf
# rV64EbTI40zqegPZdA+sXCmbcZy6okx/SjwsusWRItFA3DE8MORZeFb6BmzBtqKJ
# 7l939bbKBy2jvxcJI98Va95Q5JnlKor3m0E7xpMeYRriWklUPsetMSf2NvUQa/E5
# vVyefQIwggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0GCSqGSIb3DQEB
# DAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLTAr
# BgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290IFI0NjAeFw0y
# MTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIB
# gQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEeAEZGbEN4QMgC
# sJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGVoYW3haT29PST
# ahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk8XL/tfCK6cPu
# YHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzhP06ShdnRqtZl
# V59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41aHU5pledi9lC
# BbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+ac7IH60B+Ja7
# TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/2yGgW+ufflcZ
# /ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9gRvd+QOfdRrJZ
# b1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4Xm
# MB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8BAf8EBAMCAYYw
# EgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDAzAbBgNVHSAE
# FDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyGOmh0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nUm9vdFI0Ni5j
# cmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8vY3J0LnNlY3Rp
# Z28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYucDdjMCMGCCsG
# AQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOC
# AgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW4HL9hcpkOTz5
# jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIhrCymlaS98+Qp
# oBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYpmlG+vd0xqlqd
# 099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7DcML4iTAWS+MVX
# eNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yELg09Jlo8BMe8
# 0jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRzNyE6bxuSKcut
# isqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw4rg3sTCggkHS
# RqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6E52z1NZJ6ctu
# MFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6xWls/GDnVNue
# KjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZtg0y9ShQoPzmC
# cn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2RwwggY/MIIEp6ADAgEC
# AhA2nIGBal4DSIUR8kAAOO1LMA0GCSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdC
# MRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVi
# bGljIENvZGUgU2lnbmluZyBDQSBSMzYwHhcNMjMxMjAxMDAwMDAwWhcNMjYxMTMw
# MjM1OTU5WjBWMQswCQYDVQQGEwJERTEbMBkGA1UECAwSQmFkZW4tV8O8cnR0ZW1i
# ZXJnMRQwEgYDVQQKDAtDQU5DT00gR21iSDEUMBIGA1UEAwwLQ0FOQ09NIEdtYkgw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDArSw9LTzTaw9/KvfNlE0S
# Fvf9qcOQ/lXAPYkhX3xVx2L1/MrBosbFLHvJJ6iWD2z4717VRgN/f4woMx1ZZ+k5
# so33kL6rTKliDv1C91rd1aI4NGxwVMg0wOLKA0UDN0oR8/WgKES3QBJYq//zi3+V
# vARTml1dpOcQje3dXPcRY1m8zo/ObmB8pPr8pQAYTgmZuYtj+IZe1G96HnEMaglZ
# T7Ko9CiuCT6MoUuB7P2w+JMpRRUSW7sEk1eex3+g6iL/NcthpF3YqWCgWwNHys0b
# oYNaeQuH2BC4o4Lv2yXdR8nK1SuVaYelCd4b5hICwL3as6+z8Y2J0HEY3focyhtZ
# sHtahMk8STDcjvgWeap79Qv5jktC7pPdKxv1aI/U0d0V/vzQv+aJ0ySIrHbGMrwG
# MlqSRULednSIxm0YqUnay3C+w+9g+cF9A3TcAGHJwiBZrAvYexcMz4yC/ZpEIVpd
# nfzJZQkzqUO0AuBg0qoXbTGb67ZZwHoJKiZjLXRYuWK3Swzei0wj1VWqvmMgEVC9
# zAI3pC0kYXq9OUvFhEYyQ5iCgbluVK1Q60BwsuP1L1z81PH+0gfPpIln1cg35Dp5
# xL6O+wOC1J5GmlqfAzbwVYblBjAo8J0oZgBGHA1akmFBEINDTHxdDgNh+f/2Jop9
# z/gm4uXWBzSSRVw6TbCfaQIDAQABo4IBiTCCAYUwHwYDVR0jBBgwFoAUDyrLIIco
# uOxvSK4rVKYpqhekzQwwHQYDVR0OBBYEFDtSj69rtLzxAlC2x8F+gn+1H8w/MA4G
# A1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
# MEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUwIwYIKwYBBQUHAgEWF2h0dHBz
# Oi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBJBgNVHR8EQjBAMD6gPKA6hjho
# dHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NB
# UjM2LmNybDB5BggrBgEFBQcBAQRtMGswRAYIKwYBBQUHMAKGOGh0dHA6Ly9jcnQu
# c2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nQ0FSMzYuY3J0MCMG
# CCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwF
# AAOCAYEAJjsMqKSeh1xy940zL3vIr3V1dl2f7Qp49GXD+7pXXfHABvw/y7AZFLCT
# PoHmkV71A3VBbQ9xc2Avzd1lQOXolYC/SuO7CWPPLpcK7j15YxurHKhxufC50f6E
# Cmg9ziRMtKIvbjJRX5FDXFfHsEcNJ+3gX2jgaWxB0UgdqE/p4ionURoMze8yHHrQ
# F59br2lAFwr4IQzVjDsh/2Kki8QugddiA506lbDHiVaxKi29hVQ/EXXzWpUzJf8r
# gCaYjKSEXAYq264QDPnEdvq7oTjXG/25VCodN/H7EpHaHv10kSvUIL8QV4ZUKPab
# 631VFavfLaKHgQhYjzWdXe71qcvuLdaSObTbzWlwK9/l9G3cj5HYpnyaAba1nhIe
# MiWiiAc0tGEfS9DVfnbdr1zn3U/PiAVbsoaXx4RhlvIBnMOke2HIOcpVgWraHDN1
# j73vyOagdJBbD1alLFuFVoVhMDQ6OHQcSUBt/xuZ/SUsauguKfdmlPfYwGsdA/f1
# mAks8ZrRMIIGXTCCBMWgAwIBAgIQOlJqLITOVeYdZfzMEtjpiTANBgkqhkiG9w0B
# AQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSww
# KgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjAeFw0y
# NDAxMTUwMDAwMDBaFw0zNTA0MTQyMzU5NTlaMG4xCzAJBgNVBAYTAkdCMRMwEQYD
# VQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxMDAuBgNV
# BAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgU2lnbmVyIFIzNTCCAiIw
# DQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAI3RZ/TBSJu9/ThJOk1hgZvD2NxF
# pWEENo0GnuOYloD11BlbmKCGtcY0xiMrsN7LlEgcyoshtP3P2J/vneZhuiMmspY7
# hk/Q3l0FPZPBllo9vwT6GpoNnxXLZz7HU2ITBsTNOs9fhbdAWr/Mm8MNtYov32os
# vjYYlDNfefnBajrQqSV8Wf5ZvbaY5lZhKqQJUaXxpi4TXZKohLgxU7g9RrFd477j
# 7jxilCU2ptz+d1OCzNFAsXgyPEM+NEMPUz2q+ktNlxMZXPF9WLIhOhE3E8/oNSJk
# NTqhcBGsbDI/1qCU9fBhuSojZ0u5/1+IjMG6AINyI6XLxM8OAGQmaMB8gs2IZxUT
# OD7jTFR2HE1xoL7qvSO4+JHtvNceHu//dGeVm5Pdkay3Et+YTt9EwAXBsd0PPmC0
# cuqNJNcOI0XnwjE+2+Zk8bauVz5ir7YHz7mlj5Bmf7W8SJ8jQwO2IDoHHFC46ePg
# +eoNors0QrC0PWnOgDeMkW6gmLBtq3CEOSDU8iNicwNsNb7ABz0W1E3qlSw7jTmN
# oGCKCgVkLD2FaMs2qAVVOjuUxvmtWMn1pIFVUvZ1yrPIVbYt1aTld2nrmh544Auh
# 3tgggy/WluoLXlHtAJgvFwrVsKXj8ekFt0TmaPL0lHvQEe5jHbufhc05lvCtdwbf
# Bl/2ARSTuy1s8CgFAgMBAAGjggGOMIIBijAfBgNVHSMEGDAWgBRfWO1MMXqiYUKN
# UoC6s2GXGaIymzAdBgNVHQ4EFgQUaO+kMklptlI4HepDOSz0FGqeDIUwDgYDVR0P
# AQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgw
# SgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6
# Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEoGA1UdHwRDMEEwP6A9oDuGOWh0
# dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ0NB
# UjM2LmNybDB6BggrBgEFBQcBAQRuMGwwRQYIKwYBBQUHMAKGOWh0dHA6Ly9jcnQu
# c2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ0NBUjM2LmNydDAj
# BggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEM
# BQADggGBALDcLsn6TzZMii/2yU/V7xhPH58Oxr/+EnrZjpIyvYTz2u/zbL+fzB7l
# brPml8ERajOVbudan6x08J1RMXD9hByq+yEfpv1G+z2pmnln5XucfA9MfzLMrCAr
# NNMbUjVcRcsAr18eeZeloN5V4jwrovDeLOdZl0tB7fOX5F6N2rmXaNTuJR8yS2F+
# EWaL5VVg+RH8FelXtRvVDLJZ5uqSNIckdGa/eUFhtDKTTz9LtOUh46v2JD5Q3nt8
# mDhAjTKp2fo/KJ6FLWdKAvApGzjpPwDqFeJKf+kJdoBKd2zQuwzk5Wgph9uA46VY
# K8p/BTJJahKCuGdyKFIFfEfakC4NXa+vwY4IRp49lzQPLo7WticqMaaqb8hE2QmC
# FIyLOvWIg4837bd+60FcCGbHwmL/g1ObIf0rRS9ceK4DY9rfBnHFH2v1d4hRVvZX
# yCVlrL7ZQuVzjjkLMK9VJlXTVkHpuC8K5S4HHTv2AJx6mOdkMJwS4gLlJ7gXrIVp
# nxG+aIniGDCCBoIwggRqoAMCAQICEDbCsL18Gzrno7PdNsvJdWgwDQYJKoZIhvcN
# AQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYD
# VQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3Jr
# MS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5
# MB4XDTIxMDMyMjAwMDAwMFoXDTM4MDExODIzNTk1OVowVzELMAkGA1UEBhMCR0Ix
# GDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJs
# aWMgVGltZSBTdGFtcGluZyBSb290IFI0NjCCAiIwDQYJKoZIhvcNAQEBBQADggIP
# ADCCAgoCggIBAIid2LlFZ50d3ei5JoGaVFTAfEkFm8xaFQ/ZlBBEtEFAgXcUmanU
# 5HYsyAhTXiDQkiUvpVdYqZ1uYoZEMgtHES1l1Cc6HaqZzEbOOp6YiTx63ywTon43
# 4aXVydmhx7Dx4IBrAou7hNGsKioIBPy5GMN7KmgYmuu4f92sKKjbxqohUSfjk1mJ
# lAjthgF7Hjx4vvyVDQGsd5KarLW5d73E3ThobSkob2SL48LpUR/O627pDchxll+b
# TSv1gASn/hp6IuHJorEu6EopoB1CNFp/+HpTXeNARXUmdRMKbnXWflq+/g36NJXB
# 35ZvxQw6zid61qmrlD/IbKJA6COw/8lFSPQwBP1ityZdwuCysCKZ9ZjczMqbUcLF
# yq6KdOpuzVDR3ZUwxDKL1wCAxgL2Mpz7eZbrb/JWXiOcNzDpQsmwGQ6Stw8tTCqP
# umhLRPb7YkzM8/6NnWH3T9ClmcGSF22LEyJYNWCHrQqYubNeKolzqUbCqhSqmr/U
# dUeb49zYHr7ALL8bAJyPDmubNqMtuaobKASBqP84uhqcRY/pjnYd+V5/dcu9ieER
# jiRKKsxCG1t6tG9oj7liwPddXEcYGOUiWLm742st50jGwTzxbMpepmOP1mLnJskv
# ZaN5e45NuzAHteORlsSuDt5t4BBRCJL+5EZnnw0ezntk9R8QJyAkL6/bAgMBAAGj
# ggEWMIIBEjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4E
# FgQU9ndq3T/9ARP/FqFsggIv0Ao9FCUwDgYDVR0PAQH/BAQDAgGGMA8GA1UdEwEB
# /wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAA
# MFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VS
# VHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDA1BggrBgEFBQcBAQQp
# MCcwJQYIKwYBBQUHMAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZI
# hvcNAQEMBQADggIBAA6+ZUHtaES45aHF1BGH5Lc7JYzrftrIF5Ht2PFDxKKFOct/
# awAEWgHQMVHol9ZLSyd/pYMbaC0IZ+XBW9xhdkkmUV/KbUOiL7g98M/yzRyqUOZ1
# /IY7Ay0YbMniIibJrPcgFp73WDnRDKtVutShPSZQZAdtFwXnuiWl8eFARK3PmLqE
# m9UsVX+55DbVIz33Mbhba0HUTEYv3yJ1fwKGxPBsP/MgTECimh7eXomvMm0/GPxX
# 2uhwCcs/YLxDnBdVVlxvDjHjO1cuwbOpkiJGHmLXXVNbsdXUC2xBrq9fLrfe8IBs
# A4hopwsCj8hTuwKXJlSTrZcPRVSccP5i9U28gZ7OMzoJGlxZ5384OKm0r568Mo9T
# YrqzKeKZgFo0fj2/0iHbj55hc20jfxvK3mQi+H7xpbzxZOFGm/yVQkpo+ffv5gdh
# p+hv1GDsvJOtJinJmgGbBFZIThbqI+MHvAmMmkfb3fTxmSkop2mSJL1Y2x/955S2
# 9Gu0gSJIkc3z30vU/iXrMpWx2tS7UVfVP+5tKuzGtgkP7d/doqDrLF1u6Ci3TpjA
# ZdeLLlRQZm867eVeXED58LXd1Dk6UvaAhvmWYXoiLz4JA5gPBcz7J311uahxCweN
# xE+xxxR3kT0WKzASo5G/PyDez6NHdIUKBeE3jDPs2ACc6CkJ1Sji4PKWVT0/MYIG
# QDCCBjwCAQEwaDBUMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSswKQYDVQQDEyJTZWN0aWdvIFB1YmxpYyBDb2RlIFNpZ25pbmcgQ0EgUjM2
# AhA2nIGBal4DSIUR8kAAOO1LMA0GCWCGSAFlAwQCAQUAoIGEMBgGCisGAQQBgjcC
# AQwxCjAIoAKAAKECgAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
# BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIKp/N9aQDSYD
# 3SH7PUs70xyaOdj9jFPbJHZ1xIR/D25JMA0GCSqGSIb3DQEBAQUABIICAGXTyLYo
# fnnw+E8qjsB6rjChB7Nv3bwgXLujJ04suGvGxX/gurEmMLs9o8a5xyNwFUPD07ze
# II22woFOULc6tuDhr4923+842U6QlLSK4mBXhjE48p9w0X2/IhzfZuMT1g04fgmS
# lSkfr3iiyvAl/LqRWkRrT5E83KopRlwrc2xN0o7NdnxgjrvAgBWn6ZkG8Pz22iUJ
# SN8QObEcRYIyCa+E7gwVa/Ir2NQbX//4DAGhQimFDNjKNTCKWSCMOnANoh52yk2t
# V8MpL7HhCxTwZx8auBemScP1/rzjhBxoQxz5WnUUKhwvGqiNVFUabS9A1o9wkI3b
# ydRdgvNsSfCquJ/c1koAe8JEm1BmtynNe0Hoijv325+w87eLtDOOQe4Ti0Hm3eNw
# j5Wf+clSBCDLXxT7OP06tN4iOK6htgxL/IzaBVOm8+jKlRCTMSHSJkGNwYrp62ro
# ICK/Rrs86P0ZZPvq1gv2NcUn2+Jp6BGh2e5faOe4d7v1HeE9hGE5/NRkcsNWg8PR
# 0Zt0Z30XIT0yDr0TR1rAMFHBCOb9DMId5Ps+08bXvMhrHfYn8L5oS5MiWVxw57F/
# YHfSIE7jBOBc7CUL8DQBtJSGXBqIk0DGz78hhYARcK05p7vtX4Vz/2Y2GE/h893t
# z0BXb3xiqgODZvFI8+F6efTof44t4bw7hce6oYIDIjCCAx4GCSqGSIb3DQEJBjGC
# Aw8wggMLAgEBMGkwVTELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEsMCoGA1UEAxMjU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBS
# MzYCEDpSaiyEzlXmHWX8zBLY6YkwDQYJYIZIAWUDBAICBQCgeTAYBgkqhkiG9w0B
# CQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0yNDEwMjIxNTMzMTFaMD8G
# CSqGSIb3DQEJBDEyBDAhJM6/+e/3dYrfmYEBhOttomepwcYwMTl9NR5CaBPuEDtf
# GR7EmhQTUI56MdJfl1EwDQYJKoZIhvcNAQEBBQAEggIAHsE3sjyTTxkSHLEQL+tX
# kDSJVTbm2/py4/5qiLrm/ur/5zwi5elIIw/ABIBJL94Fep11s4YVEpdg3Z4SKrws
# IhIBlOfPmvrJ02lCwt628F2uTkEzeVzrq14s+9ujtmaCbtaa8e7dDJHKt27Hf799
# WJNyRKz+fF8uz86LRczNprVhF8Xj5WiIQGEj44Cc7qgyzEg9gIS36Db5SszoV5/O
# /qQCoCAiiPytQtb5TKXGMmA8lVmQCJ/UxLuUMzHLUmP1IadC7T41EpIPft+xT1W7
# k/pROOioU+kWcBiY0Fjm9uGrp1MId5gXj3/oph3XpIN8kzecRICsOWoDbZ/+kBdu
# QVJT31tRA4dz0148Rz4wNdsWjrZQzV0/3g5mp8BAFIYQAgDUT/qL4Z546vY7+FCg
# H/vQx/gAzM+OYtqCEvMloURJUT2e+rrzFaNqz6dlVfUrrHpSslD5wpoS8NZecqC9
# 13GozJWLtwWLBOsKnmpspMmRG43jPGtoRzHcmgKRAOUU+P3q3ppuaLxSxsIAc9rh
# 1hztTQgBLf8duGiHIkzb+MsZx0cIB9UF4HxMpaOcTFz89HuyuUKggzOwL+BITZhi
# 9EvX3w7mHaxeeVqp8R788N3IhT6Na/2YkiLaCLwhI8FdFVF8j7Mv6WNRYx6dFBAA
# NTyDG00h7qnbeDNRl4vR/uk=
# SIG # End signature block
